import argparse, json
from pathlib import Path
import numpy as np

def inf_norm(A: np.ndarray) -> float:
    return float(np.max(np.sum(np.abs(A), axis=1)))

def main():
    ap = argparse.ArgumentParser(description="Invertibility/conditioning diagnostics for kernels.")
    ap.add_argument("--kernels", default="results", help="Folder with kernels/*.npy and kernel_index.json")
    ap.add_argument("--out", default="results/kernel_inv_checks.csv", help="Output CSV path")
    ap.add_argument("--tol", type=float, default=1e-12, help="Tolerance for ||M M^{-1} - I||_inf")
    args = ap.parse_args()

    idx = json.loads(Path(args.kernels, "kernel_index.json").read_text(encoding="utf-8"))
    lines = ["n,cond_number,max_mm_inv_residual,tol,pass\n"]
    worst_cond = 0.0
    worst_resid = 0.0
    for ent in idx:
        n = float(ent["n"])
        M = np.load(ent["path"])
        I = np.eye(M.shape[0])
        try:
            Minv = np.linalg.inv(M)
        except np.linalg.LinAlgError:
            cond = float('inf'); resid = float('inf'); ok = False
        else:
            cond = float(np.linalg.cond(M))
            resid = inf_norm(M.dot(Minv) - I)
            ok = resid <= args.tol and np.isfinite(cond)
        worst_cond = max(worst_cond, cond if np.isfinite(cond) else 1e300)
        worst_resid = max(worst_resid, resid if np.isfinite(resid) else 1e300)
        lines.append(f"{n},{cond},{resid},{args.tol},{int(ok)}\n")

    Path(args.out).parent.mkdir(parents=True, exist_ok=True)
    Path(args.out).write_text("".join(lines), encoding="utf-8")
    print(f"Wrote {args.out}. worst cond={worst_cond}, worst residual={worst_resid}")
    if worst_resid > args.tol:
        print(f"WARNING: worst residual {worst_resid} > tol {args.tol}")

if __name__ == "__main__":
    main()
